/*******************************************************************************
 * Copyright (C) 2002, 2003
 * ingenieurbuero fuer innovative informationstechnik (iiit)
 * Dipl.-Ing. Joerg Beckmann, Dortmund, Germany
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * or look at http://www.gnu.org/copleft/lesser.html.
 *
 * version $Id: GenericCacheTest.java,v 1.4 2003/03/12 20:15:11 joerg Exp $
 ******************************************************************************/

package de.iiit.cache;

import java.util.*;
import junit.framework.*;

/** 
 * Implements a test suite for GenericCache
 *
 * @version $Revision: 1.4 $ $Date: 2003/03/12 20:15:11 $
 */
public class GenericCacheTest extends TestCase
{
    /** CVS Version Tag */
    private static final String vcid = "$Id: GenericCacheTest.java,v 1.4 2003/03/12 20:15:11 joerg Exp $";

    private static int LOOPS = 100000;
    
    private GenericCache cache = null;
    private int task = 0;
    
    public GenericCacheTest(java.lang.String testName)
    {
        super(testName);
    }
    
    public static void main(java.lang.String[] args)
    {
        junit.textui.TestRunner.run(suite());
    }
    
    public static Test suite()
    {
        TestSuite suite = new TestSuite(GenericCacheTest.class);
        
        return suite;
    }
    
    private void setup()
    {
        cache = new GenericCache();
    }
    
    private void reset()
    {
        cache.clear();
        cache = null;
    }
    
    private static void sleep(int ms)
    {
        try
        {
            Thread.sleep(ms);
        }
        catch (Exception e)
        {
            // do nothing 
        }
    }
    
    private void wait(Thread thread)
    {
        // Wait until thread has finished
        while (thread.isAlive())
        {
            sleep(10);
        }
    }
    
    public void testAddElement()
    {
        System.out.println("GenericCacheTest.testAddElement");
        
        setup();
        
        long t = System.currentTimeMillis();
        
        for (int i = 0; i < LOOPS; i++)
        {
            Object obj = new Object();
            String name = "Obj-" + i;
            
            cache.addElement(name, obj);
        }
        
        t = System.currentTimeMillis() - t;
        System.out.println(LOOPS + " elements inserted within " + t + " ms");        
                
        Assert.assertEquals("Number of elements", LOOPS, cache.size());
        
        reset();
    }
    
    public void testMultiThreadedAddElement()
    {
        System.out.println("GenericCacheTest.testMultiThreadedAddElement");
        
        setup();
        
        TestThread thread = new TestThread();
        
        thread.start();
        
        long t = System.currentTimeMillis();
        
        for (int i = 0; i < LOOPS; i++)
        {
            Object obj = new Object();
            String name = "Main-" + i;
            
            cache.addElement(name, obj);
        }
        
        t = System.currentTimeMillis() - t;
        System.out.println(LOOPS + " elements inserted within " + t + " ms");        
                
        wait(thread);
        
        Assert.assertEquals("Number of elements", 2 * LOOPS, cache.size());

        reset();
    }
    
    public void testMultiThreadedAddGetElement()
    {
        System.out.println("GenericCacheTest.testMultiThreadedAddGetElement");
        
        setup();
        
        TestThread thread = new TestThread();
        
        thread.start();
        
        long t = System.currentTimeMillis();
        
        for (int i = 0; i < LOOPS; i++)
        {
            while (thread.isAlive() && cache.size() <= i)
            {
                sleep(1);
            }
            
            String name = "Thread-" + i;
            
            try 
            {
                Object obj = cache.getElement(name);
            }
            catch (CacheFaultException cfe)
            {
                fail("Object no. " + i + " not found");
            }
        }
        
        t = System.currentTimeMillis() - t;
        System.out.println(LOOPS + " elements read within " + t + " ms");        
                
        wait(thread);
        
        Assert.assertEquals("Number of elements", LOOPS, cache.size());

        reset();
    }
    
    public void testMultiThreadedAddRemoveElement()
    {
        System.out.println("GenericCacheTest.testMultiThreadedAddRemoveElement");
        
        setup();
        
        TestThread thread = new TestThread();
        
        thread.start();
        
        long t = System.currentTimeMillis();
        
        for (int i = 0; i < LOOPS; i++)
        {
            while (thread.isAlive() && cache.size() == 0)
            {
                sleep(1);
            }
            
            String name = "Thread-" + i;
            
            cache.removeElement(name);
        }
        
        t = System.currentTimeMillis() - t;
        System.out.println(LOOPS + " elements removed within " + t + " ms");        
                
        wait(thread);
        
        Assert.assertEquals("Number of elements", 0, cache.size());

        reset();
    }
    
    public void testMultiThreadedAddElementGetKeys()
    {
        System.out.println("GenericCacheTest.testMultiThreadedAddElementGetKeys");
        
        setup();
                
        TestThread thread = new TestThread();
        
        thread.start();
        
        long t = System.currentTimeMillis();
        
        int cnt = 0;
        while (thread.isAlive())
        {
            cnt++;
            
            Enumeration e = cache.getKeys();
            
            while (e.hasMoreElements())
            {
                e.nextElement();
            }
        }
        
        t = System.currentTimeMillis() - t;
        System.out.println(cnt + " loops done within " + t + " ms");        
                
        Assert.assertEquals("Number of elements", LOOPS, cache.size());

        reset();
    }
    
    public void testMultiThreadedRemoveElementGetKeys()
    {
        System.out.println("GenericCacheTest.testMultiThreadedRemoveElementGetKeys");
        
        setup();
        
        long t = System.currentTimeMillis();
        
        for (int i = 0; i < LOOPS; i++)
        {
            Object obj = new Object();
            String name = "Obj-" + i;
            
            cache.addElement(name, obj);
        }
        
        t = System.currentTimeMillis() - t;
        System.out.println(LOOPS + " elements inserted within " + t + " ms");        

        task = 1;
        TestThread thread = new TestThread();
        
        thread.start();
        
        t = System.currentTimeMillis();
        
        while (thread.isAlive() && cache.size() == 0)
        {
            sleep(1);
        }
   
        int cnt = 0;
        int dcnt = 0;
        while (thread.isAlive())
        {
            cnt++;
            
            Enumeration e = cache.getKeys();
            
            while (e.hasMoreElements())
            {
                try 
                {
                    Object obj = cache.getElement((String) e.nextElement());
                }
                catch (CacheFaultException cfe)
                {
                    dcnt++;
                }
            }
        }
        
        t = System.currentTimeMillis() - t;
        System.out.println(cnt + " loops done within " + t + " ms, " + dcnt + " elements where already deleted");        
                
        Assert.assertEquals("Number of elements", 0, cache.size());

        reset();
    }
    
    private class TestThread extends Thread
    {
        public void run()
        {
            long t;
            
            switch (task)
            {
                case 0:
                    // Multi-threaded add elements

                    t = System.currentTimeMillis();

                    for (int i = 0; i < LOOPS; i++)
                    {
                        Object obj = new Object();
                        String name = "Thread-" + i;

                        cache.addElement(name, obj);
                    }

                    t = System.currentTimeMillis() - t;
                    System.out.println(LOOPS + " elements inserted within " + t + " ms"); 

                    break;

                case 1:
                    // Multi-threaded remove elements

                    t = System.currentTimeMillis();

                    for (int i = 0; i < LOOPS; i++)
                    {
                        while (cache.size() == 0)
                        {
                            GenericCacheTest.sleep(1);
                        }

                        String name = "Obj-" + i;

                        cache.removeElement(name);
                    }
                    t = System.currentTimeMillis() - t;
                    System.out.println(LOOPS + " elements removed within " + t + " ms"); 

                    break;

                default:
                    fail("Unknown task <" + task + ">");
                    break;                
            }
        }
        
    }
    
}

/**
 * $Log: GenericCacheTest.java,v $
 * Revision 1.4  2003/03/12 20:15:11  joerg
 * Unused code removed
 *
 * Revision 1.3  2003/03/12 19:40:03  joerg
 * Error in clear() removed
 *
 * Revision 1.2  2003/03/12 13:22:18  joerg
 * vcid added.
 *
 * Revision 1.1  2003/03/12 13:20:58  joerg
 * 1st version of unit tests
 *
 */
