/*******************************************************************************
 * Copyright (C) 2002, 2003
 * ingenieurbuero fuer innovative informationstechnik (iiit)
 * Dipl.-Ing. Joerg Beckmann, Dortmund, Germany
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * or look at http://www.gnu.org/licenses/lgpl.html.
 *
 * version $Id: Configuration.java,v 1.15 2003/04/02 15:58:01 joerg Exp $
 ******************************************************************************/

package de.iiit.xmlconfig;

import de.iiit.util.*;

import java.io.*;
import java.util.*;

import org.apache.xerces.parsers.*;
import org.apache.log4j.Logger;

import org.xml.sax.*;
import org.xml.sax.helpers.*;

import org.apache.log4j.Logger;

/** Represents the configuration read by ConfigFileReader.
 * 
 * @version $Revision: 1.15 $ $Date: 2003/04/02 15:58:01 $
 */
public class Configuration
{
    /** CVS Version Tag */
    private static final String vcid = "$Id: Configuration.java,v 1.15 2003/04/02 15:58:01 joerg Exp $";
    
    private String      name;
    private Hashtable   attributes  = new Hashtable();
    private Vector      subConfigs  = new Vector();
    
    /** Creates a new instance of Configuration */
    private Configuration()
    {
    }
    
    /** Creates a new instance of Configuration
     * @param name The name of the configuration
     */    
    protected Configuration(String name)
    {
        this.name = name;
    }
    
    /** Loads a configuration from a XML file.
     * @param filename the full pathname of the file to read.
     * @throws FileNotFoundException if the named file does not exist, is a directory rather than a regular file, or
     * for some other reason cannot be opened for reading.
     * @throws IOException if an I/O error occurs.
     * @throws SAXException any SAX exception, possibly wrapping another exception.
     * @return the read configuration or null if the file is empty.
     */    
    public static Configuration load(String filename) 
    throws FileNotFoundException, IOException, SAXException    
    {
        return new Reader().readConfigFile(filename);
    }
    
    /** Saves the XML representation of the current configuration to a file.
     * @param filename the full pathname of the file to write.
     * @throws IOException if an I/O error occurs.
     */    
    public void save(String filename)
    throws IOException
    {
        StringBuffer buffer = new StringBuffer("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n\n");
        
        buffer.append(toXML());

        FileWriter fw = new FileWriter(filename);
        fw.write(buffer.toString());
        fw.close();
    }
    
    private String indentSpace(int indent)
    {
        String space = "";
        
        for (int i = 0; i < indent; i++)
            space += "  ";
        
        return space;
    }
    
    private StringBuffer toXML(int indent)
    {
        StringBuffer buffer = new StringBuffer();
        String       space  = indentSpace(indent);
        boolean      attributeEmptyFlag = attributes.isEmpty();
        boolean      subConfigEmptyFlag = subConfigs.isEmpty();
        
        if (attributeEmptyFlag && subConfigEmptyFlag)
            buffer.append(space + "<" + name + "/>\n\n");
        else if (attributeEmptyFlag)
            buffer.append(space + "<" + name + ">\n");
        else
        {
            buffer.append(space + "<" + name + "\n");
            
            Enumeration keys = attributes.keys();
        
            while(keys.hasMoreElements())
            {
                String k = (String) keys.nextElement();
                String v = (String) getAttribute(k);
                buffer.append(space + "  " + k + "\t= \"" + v + "\"\n");
            }
            
            if (subConfigEmptyFlag)
                buffer.append(space + "/>\n\n");
            else
                buffer.append(space + ">\n");                
        }

        if (! subConfigEmptyFlag)
        {
            Iterator i = subConfigs.iterator();
        
            while (i.hasNext())
                buffer.append(((Configuration) i.next()).toXML(indent+1));

            buffer.append(space + "</" + getName() + ">\n\n");
        }
        
        return buffer;
    }
    
    /** Creates a XML representation of the current configuration
     * @return the XML data.
     */    
    public StringBuffer toXML()
    {
        return toXML(0);
    }
    
    private void printLine(Logger logger, String msg)
    {
        if (logger != null)
            logger.debug(msg);
        else
            System.out.println(msg);
    }
    
    private void printConfig(Logger logger, int indent)
    {
        String space = indentSpace(indent);
        
        printLine(logger, space + "Sub-Config: " + getName());
        
        for (Enumeration keys = getAttributeKeys(); keys.hasMoreElements(); )
        {
            String k = (String) keys.nextElement();
            String v = (String) getAttribute(k);
            printLine(logger, space + "  " + k + " => " + v);
        }

        Vector v = getSubConfigurations();
        
        int iMax = v.size();
        
        for (int i = 0; i < iMax; i++)
        {
            ((Configuration) v.get(i)).printConfig(logger, indent+1);
        }     
    }

    /** Print the current configuration to an already initialized Log4J-<I>Logger</I> with
     * level <I>DEBUG</I>.
     */    
    public void logConfig()
    {
        printConfig(Logger.getLogger(this.getClass()), 0);
    }
    
    /** Print the current configuration to <I>System.out()</I>. */    
    public void printConfig()
    {
        printConfig(null, 0);
    }
    
    /** Set the name of the current configuration
     * @param name The name
     */    
    protected void setName(String name)
    {
        this.name = name;
    }
    
    /** Adds a subconfiguration
     * @param config The configuration to add
     */    
    public void addSubConfiguration(Configuration config)
    {
        subConfigs.add(config);
    }
    
    /** Add a single attribute to the configuration
     * @param key The name of the attribute
     * @param value The value of the attribute
     */    
    public void addAttribute(String key, String value)
    {
        attributes.put(key.toLowerCase(), value);
    }
    
    /** Get the name of the current configuration
     * @return The name of the current configuration
     */    
    public String getName()
    {
        return name;
    }
    
    /** Get all sub-configurations of the current configuration
     * @return A <I>Vector</I> of <I>Configuration</I> objects or <B>null</B> if there is no subconfiguration
     */    
    public Vector getSubConfigurations()
    {
        return subConfigs;
    }
    
    /** Get all sub-configurations of the current configuration with a given name
     * @param name The name of the sub-configuration to retrieve
     * @return A <I>Vector</I> with the retrieved <I>Configuration</I> Object or <B>null</B> if there is no
     * sub-configuration with the given <I>name</I>.
     */    
    public Vector getSubConfigurations(String name)
    {
        Vector v = getSubConfigurations();
        Vector result = new Vector();
        
        name = name.toLowerCase();
        
        int iMax = v.size();
        
        for (int i = 0; i < iMax; i++)
        {
            Configuration c = (Configuration) v.get(i);
            if (c.getName().equals(name))
                result.add(c);
        }     
        
        return result.size() > 0 ? result : null;
    }
    
    /** Retrieve a single sub-configuration with a given name. If there is more than one
     * Block with this name only the first of them is returned.
     * @param name The name of the sub-configuration
     * @return The sub-configuration or null if there is no one with the given name.
     */    
    public Configuration getSubConfiguration(String name)
    {
        Configuration result = null;

        Vector v = getSubConfigurations(name);
        if (v != null)
        {
            result = (Configuration) v.get(0);
        }

        return result;
    }
    
    /** Get the <I>Hashtable</I> containig all attributes of the current configuration
     * @return A <I>HashTable</I> with all attributes
     */    
    public Hashtable getAttributes()
    {
        return attributes;
    }
    
    /** Retrieve an attribute of the current configuration with a given name
     * @param key The name of the attribute to retrieve
     * @return The attribute with the given name or <B>null</B> if there is no attribute with the
     * given name.
     */    
    public String getAttribute(String key)
    {
        return (String) attributes.get(key.toLowerCase());
    }
    
    /** Retrieve an integer attribute of the current configuration with a given name
     * @param key The name of the attribute to retrieve
     * @return The value of attribute with the given name. If there is no attribute with the
     * given name <B>0</B> is returned.
     */    
    public int getIntAttribute(String key)
    {
        String s = (String) attributes.get(key.toLowerCase());
        
        if (s != null)
            return Integer.parseInt(s);
        else
            return 0;
    }
    
    /** Retrieve an boolean attribute of the current configuration with a given name
     * @param key The name of the attribute to retrieve
     * @return The value of attribute with the given name. If there is no attribute with the
     * given name, <B>false</B> is returned.
     */    
    public boolean getBooleanAttribute(String key)
    {        
        String s = (String) attributes.get(key.toLowerCase());
        
        return (s != null && (s.equalsIgnoreCase("TRUE") || s.equalsIgnoreCase("YES") || s.equalsIgnoreCase("1")));
    }
    
    /** Retrieve an attribute from the current configuration with a given name
     * @param key The name of the attribute to retrieve
     * @param defaultValue The value to return if the is no attribute with the given name
     * @return The attribute with the given name or <I>defaultValue</I> if there is no attribute with the
     * given name.
     */    
    public String getAttribute(String key, String defaultValue)
    {
        String v = (String) attributes.get(key.toLowerCase());
        
        if (v == null)
            v = defaultValue;
        
        return v;
    }
    
    /** Retrieve an integer attribute of the current configuration with a given name
     * @param key The name of the attribute to retrieve
     * @param defaultValue The value to return if the is no attribute with the given name
     * @return The attribute with the given name or <I>defaultValue</I> if there is no attribute with the
     * given name.
     */    
    public int getIntAttribute(String key, int defaultValue)
    {
        return Integer.parseInt(getAttribute(key, String.valueOf(defaultValue)));
    }
    
    /** Retrieve an boolean attribute of the current configuration with a given name
     * @param key The name of the attribute to retrieve
     * @param defaultValue The value to return if the is no attribute with the given name
     * @return The attribute with the given name or <I>defaultValue</I> if there is no attribute with the
     * given name.
     */    
    public boolean getBooleanAttribute(String key, boolean defaultValue)
    {        
        String s = ((String) attributes.get(key.toLowerCase()));
        
        if (s != null)
            return (s.equalsIgnoreCase("TRUE") || s.equalsIgnoreCase("YES") || s.equals("1"));
        else
            return defaultValue;
    }
    
    /** Returns an <I>Enumeration</I> of all defined attribute names of the current configuration
     * @return An <I>Enumeration</I> of all defined attribute names of the current configuration
     */    
    public Enumeration getAttributeKeys()
    {
        return attributes.keys();
    }

    private static class Reader extends DefaultHandler
    {
        /** CVS Version Tag */
        private static final String vcid = "$Id: Configuration.java,v 1.15 2003/04/02 15:58:01 joerg Exp $";

        private Configuration rootConfig    = null;
        private Configuration currentConfig = null;

        private Stack configStack = new Stack();

        private Logger logger = Logger.getLogger(this.getClass());

        /** Creates a new instance of ConfigReader */
        public Reader()
        {
        }

        /** Implementation of the call-back method defined in Xerces' DefaultHandler
         * @param uri The Namespace URI, or the empty string if the element has no Namespace URI or if Namespace processing is not being performed.
         * @param localName The local name (without prefix), or the empty string if Namespace processing is not being performed.
         * @param qName The qualified name (with prefix), or the empty string if qualified names are not available.
         * @param attrs The attributes attached to the element. If there are no attributes, it shall be an empty Attributes object.
         */    
        public void startElement(String uri, String localName, String qName, Attributes attrs) 
        {        
            Configuration c = new Configuration(qName.toLowerCase());

            if (rootConfig == null)
            {
                rootConfig = c;
                configStack.push(c);
            }
            else
            {
                currentConfig.addSubConfiguration(c);
                configStack.push(currentConfig);
            }

            currentConfig = c;

            int iMax = attrs.getLength();

            for (int i = 0; i < iMax; i++)
            {
                c.addAttribute(attrs.getQName(i).toLowerCase(), attrs.getValue(i));
            }
        }

        /** Implementation of the call-back method defined in Xerces' DefaultHandler
         * @param uri The Namespace URI, or the empty string if the element has no Namespace URI or if Namespace processing is not being performed.
         * @param localName The local name (without prefix), or the empty string if Namespace processing is not being performed.
         * @param qName The qualified XML 1.0 name (with prefix), or the empty string if qualified names are not available.
         */    
        public void endElement(String uri, String localName, String qName ) 
        {
            currentConfig = (Configuration) configStack.pop();
        }

        /** This is the main method of this class. It tries to read the XML file named
         * <I>filename</I> and delivers it as an Object of class <I>Configuration</I>
         * @see de.iiit.xmlconfig.Configuration
         * @param filename The name of the file to read.
         * @return The read configuration or if the file was empty.
         * @throws Exception Any exception occuring while reading and parsing the file is passed through.
         */    
        public Configuration readConfigFile(String filename) 
        throws FileNotFoundException, IOException, SAXException
        {
            boolean logFlag = LogUtil.loggerIsInitialized();

            String msg = "Start reading file <" + filename + ">";
            if (logFlag)
                logger.debug(msg);
            else
                System.out.println(msg);

            rootConfig = null;
            currentConfig = null;

            SAXParser p = new SAXParser();
            p.setContentHandler(this);
            p.parse(new InputSource(new FileReader(filename)));

            msg = "Finished reading file <" + filename + ">.";
            if (logFlag)
                logger.debug(msg);
            else
                System.out.println(msg);

            return rootConfig;
        }
    }
}

/**
 * $Log: Configuration.java,v $
 * Revision 1.15  2003/04/02 15:58:01  joerg
 * New methods to load and save the configuration.
 *
 * Revision 1.14  2003/01/13 08:16:51  joerg
 * Methoden addSubconfiguration und addAttribute sind jetzt public.
 *
 * Revision 1.13  2003/01/01 21:03:49  joerg
 * Copyright-Statement aktualisiert
 *
 * Revision 1.12  2002/12/10 21:53:39  joerg
 * Copyright-Statement korrigiert.
 *
 * Revision 1.11  2002/12/07 19:58:08  joerg
 * Fehler bei null-Werten behoben
 *
 * Revision 1.10  2002/11/30 14:54:11  joerg
 * Javadoc ergaenzt
 *
 * Revision 1.9  2002/11/30 14:16:12  joerg
 * Javadoc aktualisiert,
 * Fehler in getIntAttribute beseitigt.
 *
 * Revision 1.8  2002/11/30 13:56:41  joerg
 * Fehler in Methode getBooleanAttribute() behoben
 *
 * Revision 1.7  2002/11/29 15:42:18  joerg
 * Zusaetzliche Abfrage-Methoden
 *
 * Revision 1.6  2002/11/29 14:40:55  joerg
 * Zusaetzliche Abfrage-Methoden
 *
 * Revision 1.5  2002/11/23 14:21:52  joerg
 * GPL-Statements eingefuegt
 *
 * Revision 1.4  2002/11/23 13:04:26  joerg
 * JavaDoc verbessert
 *
 * Revision 1.3  2002/11/22 16:35:48  joerg
 * Setter-Methoden sind jetzt 'protected'
 * Print-Methoden ueberarbeitet
 * JavaDoc-Kommentare eingefuegt
 *
 * Revision 1.2  2002/11/03 19:39:19  joerg
 * Methode printConfig() von Klasse ConfigReader in Klasse Configuration verschoben
 *
 * Revision 1.1  2002/11/03 15:10:19  joerg
 * Klasse 'ConfigurationElement' umbenannt in 'Configuration'
 *
 * Revision 1.1  2002/10/31 15:05:16  joerg
 * Erste Version bestehend aus Config-Reader, Logging und Expressions.
 *
 */
