/*******************************************************************************
 * Copyright (C) 2002, 2003
 * ingenieurbuero fuer innovative informationstechnik (iiit)
 * Dipl.-Ing. Joerg Beckmann, Dortmund, Germany
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * or look at http://www.gnu.org/copleft/lesser.html.
 *
 * version $Id: LogUtil.java,v 1.12 2003/04/02 09:25:53 joerg Exp $
 ******************************************************************************/

package de.iiit.util;

import de.iiit.xmlconfig.*;

import org.apache.log4j.*;

import java.io.*;

/** This class includes some methods to initialize a Log4J <I>Logger</I> 
 * @version $Revision: 1.12 $ $Date: 2003/04/02 09:25:53 $
 */
public class LogUtil {

    /** CVS Version Tag */
    private static final String vcid = "$Id: LogUtil.java,v 1.12 2003/04/02 09:25:53 joerg Exp $";

    private static String CONFIG_LOGLEVEL       = "LogLevel";
    private static String CONFIG_LOGLAYOUT      = "LogLayout";
    private static String CONFIG_LOGFILE        = "LogFile";
    private static String CONFIG_MAXFILESIZE    = "MaxFileSize";
    private static String CONFIG_MAXBACKUPFILES = "MaxBackupFiles";
    
    private static Level  defaultLevel  = Level.DEBUG;
    private static String defaultLayout = "%d [%t] %-5p %c{1} - %m%n";

    private static boolean initialized = false;
    private static Logger logger = Logger.getLogger("LogUtil");

    /** Retrieve if the <I>Logger</I> is already initialized.
     * @return <B>true</B> if the Logger is initialized or <B>false</B> if not.
     */    
    public static final boolean loggerIsInitialized()
    {
        return initialized;
    }
    
    /** Initializes the <I>Logger</I> to log all messages with an log-level of <I>level</I> or higher.
     * The output is directed to <B>STDOUT</B>.
     * @param level log all messages of this <I>level</I> or higher
     */    
    public static final void init(Level level) 
    {        
	if(!initialized) 
        {
            Logger.getRootLogger().setLevel(level);
            
            PatternLayout layout = new PatternLayout(defaultLayout);
            ConsoleAppender appender = new ConsoleAppender(layout, "System.out");
	    BasicConfigurator.configure(appender);	
	    
            initialized = true;

            logger.info("Logging system initialized, log level = <" + level.toString() + ">");
	}
    }
    
    /** Initializes the <I>Logger</I> to log all messages with an log-level of <I>level</I> or higher.
     * The output is directed to <B>STDOUT</B>.
     * @param level log all messages with a log-level named <I>level</I> or higher.
     * @param defaultLevel use this level as default if the <I>level</I> name can not be parsed.
     */    
    public static final void init(String level, Level defaultLevel) 
    {
        init(Level.toLevel(level, defaultLevel));
    }
    
    /** Initializes the <I>Logger</I> to log all messages with an log-level of <I>level</I> or higher.
     * The output is directed to a file.
     * @param level log all messages of this <I>level</I> or higher
     * @param filename the name of the file to send the log messages to. If it is null or empty
     * <CODE>System.out</CODE> will be used.
     * @param maxFileSize the maximum file size of the log file
     * @param maxBackupFiles the number of old log files  to save
     */    
    public static final void init(Level level, String filename, int maxFileSize, int maxBackupFiles)
    {
        PatternLayout layout = new PatternLayout(defaultLayout);
        
        Appender appender = null;
        
        if (filename == null || filename.equals(""))
            appender = new ConsoleAppender(layout, "System.out");
        else
        {
            try 
            {
                appender = new RollingFileAppender(layout, filename);
                ((RollingFileAppender) appender).setMaxBackupIndex(maxBackupFiles);
                ((RollingFileAppender) appender).setMaximumFileSize(maxFileSize);
            }
            catch(IOException ioe)
            {
                appender = new ConsoleAppender(layout, "System.out");

                if (initialized)
                    logger.error("Can't configure log file <" + filename + ">");
            }                
        }
        
        BasicConfigurator.resetConfiguration();
        
        Logger.getRootLogger().setLevel(level);
        BasicConfigurator.configure(appender);	

        initialized = true;
        
        logger.info("Logging system initialized, log level = <" + level.toString() + ">");            
    }
    
    /** Initializes the <I>Logger</I> to log all messages with an log-level of <I>level</I> or higher.
     * The output is directed to a file.
     * @param level log all messages of this <I>level</I> or higher
     * @param filename the name of the file to send the log messages to. If it is null or empty
     * <CODE>System.out</CODE> will be used.
     * @param maxFileSize the maximum file size of the log file
     * @param maxBackupFiles the number of old log files  to save
     */    
    public static final void init(String level, String filename, int maxFileSize, int maxBackupFiles)
    {
        init(Level.toLevel(level, defaultLevel), filename, maxFileSize, maxBackupFiles);
    }
    
    /** Initializes the <I>Logger</I> to log all messages with an log-level of <I>level</I> or higher.
     * The output is directed to a file. The maximum filesize is set to 1,000,000 and
     * a maximum of five old files is saved.
     * @param level log all messages of this <I>level</I> or higher
     * @param filename the name of the file to send the log messages to. If it is null or empty
     * <CODE>System.out</CODE> will be used.
     */    
    public static final void init(Level level, String filename)
    {
        init(level, filename, 1000000, 5);
    }
    
    /** Initializes the <I>Logger</I> to log all messages with an log-level of <I>level</I> or higher.
     * The output is directed to a file. The maximum filesize is set to 1,000,000 and
     * a maximum of five old files is saved.
     * @param level log all messages of this <I>level</I> or higher
     * @param filename the name of the file to send the log messages to. If it is null or empty
     * <CODE>System.out</CODE> will be used.
     */    
    public static final void init(String level, String filename)
    {
        init(Level.toLevel(level, defaultLevel), filename);
    }
    
    
    /** Uses a {@link Configuration } block to configure and initialize the <I>Logger</I>
     * These Attributes are read from the configuration:<br>
     *    LogLevel - log all messages of this <I>level</I> or higher<br>
     *    LogLayout - use this layout string, defaults to <CODE>"%d [%t] %-5p %c{1} - * %m%n"</CODE><br>
     *    LogFile - the full pathname of the log file to write<br>
     *    MaxFileSize - the maximum file size of the log file, default: 1,000,000
     * Bytes<br>
     *    MaxBackupFiles - the number of old log files  to save, default: 5<br>
     * @param config the configuration block to read.
     * @param defaultLevel use this log level as default
     */    
    public static final void init(Configuration config, Level defaultLevel)
    {
	if(!initialized) 
        {
            String logFile      = config.getAttribute(CONFIG_LOGFILE);
            String logLevel     = config.getAttribute(CONFIG_LOGLEVEL);
            String logLayout    = config.getAttribute(CONFIG_LOGLAYOUT, defaultLayout);

            int maxFileSize    = config.getIntAttribute(CONFIG_MAXFILESIZE, 1000000);
            int maxBackupFiles = config.getIntAttribute(CONFIG_MAXBACKUPFILES, 5); 

            if (logFile == null || logFile.equals(""))
                init(logLevel, defaultLevel);
            else
            {
                try
                {
                    Level level = Level.toLevel(logLevel, defaultLevel);
                    Logger.getRootLogger().setLevel(level);
            
                    PatternLayout layout = new PatternLayout(logLayout);

                    RollingFileAppender appender = new RollingFileAppender(layout, logFile);
                    appender.setMaxBackupIndex(maxBackupFiles);
                    appender.setMaximumFileSize(maxFileSize);

                    BasicConfigurator.configure(appender);	
	    
                    initialized = true;

                    logger.info("Logging system initialized, log level = <" + level.toString() + ">");
                    System.out.println("Logging to file <" + logFile + ">");
                }
                catch(IOException ie)
                {
                    init(logLevel, defaultLevel);
                    logger.error("Caught IOException while configuring log file <" + logFile + ">", ie);
                }
            }
        }
    }
    
    /** Uses a {@link Configuration } block to configure and initialize the <I>Logger</I>
     * These Attributes are read from the configuration:<br>
     *    LogLevel - log all messages of this <I>level</I> or higher<br>
     *    LogLayout - use this layout string, defaults to <CODE>"%d [%t] %-5p %c{1} - * %m%n"</CODE><br>
     *    LogFile - the full pathname of the log file to write<br>
     *    MaxFileSize - the maximum file size of the log file, default: 1,000,000
     * Bytes<br>
     *    MaxBackupFiles - the number of old log files  to save, default: 5<br>
     * @param config the configuration block to read.
     */    
    public static final void init(Configuration config)
    {
        init(config, defaultLevel);
    }
    
    /** Initializes the <I>Logger</I> to log all messages with an log-level of
     * <B>DEBUB</B> or higher.
     * The output is directed to <B>STDOUT</B>.
     */    
    public static final void init()
    {
        init(Level.DEBUG);
    }
}

/**
 * $Log: LogUtil.java,v $
 * Revision 1.12  2003/04/02 09:25:53  joerg
 * New variants of the init() methods
 *
 * Revision 1.11  2003/01/01 21:03:49  joerg
 * Copyright-Statement aktualisiert
 *
 * Revision 1.10  2002/12/21 21:18:10  joerg
 * Name des Log-Files wird nach System.out ausgegeben
 *
 * Revision 1.9  2002/12/10 21:53:39  joerg
 * Copyright-Statement korrigiert.
 *
 * Revision 1.8  2002/12/07 20:30:38  joerg
 * Javadoc weiter vervollstaendigt.
 *
 * Revision 1.7  2002/12/02 11:19:11  joerg
 * Neue init()-Methoden, die Konfigurationsbloecke verarbeiten.
 *
 * Revision 1.6  2002/11/30 14:54:11  joerg
 * Javadoc ergaenzt
 *
 * Revision 1.5  2002/11/23 14:21:52  joerg
 * GPL-Statements eingefuegt
 *
 * Revision 1.4  2002/11/22 16:37:06  joerg
 * JavaDoc-Kommentare eingefuegt
 *
 * Revision 1.3  2002/11/22 10:29:38  joerg
 * Neue Methode init(String level, Level defaultLevel)
 *
 * Revision 1.2  2002/11/03 20:35:42  joerg
 * Neue Methode loggerIsInitialized()
 *
 * Revision 1.1  2002/11/03 19:40:50  joerg
 * Klasse LogInit in LogUtil umbenannt
 *
 * Revision 1.1  2002/10/31 15:05:16  joerg
 * Erste Version bestehend aus Config-Reader, Logging und Expressions.
 *
 */
