/*******************************************************************************
 * Copyright (C) 2002, 2003
 * ingenieurbuero fuer innovative informationstechnik (iiit)
 * Dipl.-Ing. Joerg Beckmann, Dortmund, Germany
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * or look at http://www.gnu.org/copleft/lesser.html.
 *
 * version $Id: LdapGenericObject.java,v 1.9 2003/01/01 21:03:49 joerg Exp $
 ******************************************************************************/

package de.iiit.ldap;

import java.util.*;

import javax.naming.ldap.*;
import javax.naming.Context;
import javax.naming.*;
import javax.naming.directory.*;

import org.apache.log4j.Logger;

/** Implements a generic LDAP object to store the information requested from the
 * LDAP server
 * @version $Revision: 1.9 $ $Date: 2003/01/01 21:03:49 $
 */
public class LdapGenericObject
{
    /** CVS Version Tag */
    private static final String vcid = "$Id: LdapGenericObject.java,v 1.9 2003/01/01 21:03:49 joerg Exp $";
    
    private Logger logger = Logger.getLogger(this.getClass());
    
    private String      name        = null;
    private String      base        = null;
    private LdapContext context     = null;
    private Hashtable   attributes  = new Hashtable();
    
    /** Creates a new instance of LdapGenericObject */
    public LdapGenericObject()
    {        
    }

    /** Creates a new instance of LdapGenericObject
     * @param context The javax.naming.ldap.LdapContext containig the requested data
     * @param name the name of the object relative to the base
     * @param base the base name of the object
     */    
    public LdapGenericObject(LdapContext context, String name, String base)
    {        
        this.name    = name;
        this.base    = base;
        this.context = context;
    }

    /** Retrieves the object's name relative to the base
     * @return the relative name.
     */    
    public String getName()
    {
        return name;
    }
    
    /** Retrieves the base name of the object. In most cases this is the search base
     * used to request the objects from the LDAP server
     * @return the base name
     */    
    public String getBase()
    {
        return base;
    }
    
    /** Retrieves the absolute name of object, which is a combination of the base name
     * and the relative name
     * @return the absolute name.
     */    
    public String getAbsoluteName()
    {
        return name + "," + base;
    }
    
    /** Retrieves the attributes of the object as a HashTable containing Vectors for
     * each attribute.
     * @return the attribute hash
     */    
    public Hashtable getAttributes()
    {
        return attributes;
    }
    
    /** Retrieves a single named attribute as a Vector of Strings
     * @param key the name of attribute, eg. "cn"
     * @return the Vector of all requested attribute values
     */    
    public Vector getAttribute(String key)
    {
        return (Vector) attributes.get(key);
    }
    
    /** Add a attribute with its values to the generic object
     * @param key The name of the attribute
     * @param values the Vector containing the values of the attribute represented as Strings
     */    
    public void setAttribute(String key, Vector values)
    {
        attributes.put(key, values);
    }
    
    /** Add a attribute with its values to the generic object
     * @param key The name of the attribute
     * @param values the NamingEnumeration containing the values of the attribute represented as Strings
     * @throws NamingException if a naming exception is encountered while attempting to retrieve the elements.
     * See javax.naming.NamingException and its subclasses for the possible naming exceptions.
     */    
    public void setAttribute(String key, NamingEnumeration values) throws NamingException
    {
        Vector v = new Vector();
        
        while (values.hasMore())
            v.add(values.next());
        
        setAttribute(key, v);
    }

    /** Add a attribute with its values to the generic object
     * @param attribute The javax.naming.directory.Attribute to add to the generic object
     * @throws NamingException if a naming exception is encountered while attempting to retrieve the elements.
     * See javax.naming.NamingException and its subclasses for the possible naming exceptions.
     */    
    public void setAttribute(Attribute attribute) throws NamingException
    {
        setAttribute(attribute.getID(), attribute.getAll());
    }

    /** Add attributes with their values to the generic object
     * @param attributes The javax.naming.directory.Attributes to add to the generic object
     * @throws NamingException if a naming exception is encountered while attempting to retrieve the elements.
     * See javax.naming.NamingException and its subclasses for the possible naming exceptions.
     */    
    public void setAttributes(Attributes attributes) throws NamingException
    {
        NamingEnumeration ne = attributes.getAll();
            
        while (ne.hasMore())
        {
            setAttribute((Attribute) ne.next());
        }
    }
}

/**
 * $Log: LdapGenericObject.java,v $
 * Revision 1.9  2003/01/01 21:03:49  joerg
 * Copyright-Statement aktualisiert
 *
 * Revision 1.8  2002/12/10 21:53:39  joerg
 * Copyright-Statement korrigiert.
 *
 * Revision 1.7  2002/12/10 11:01:53  joerg
 * JavaDoc ergaenzt
 *
 * Revision 1.6  2002/12/07 20:31:07  joerg
 * Javadoc weiter vervollstaendigt.
 *
 * Revision 1.5  2002/11/30 14:53:43  joerg
 * no message
 *
 * Revision 1.4  2002/11/23 14:21:52  joerg
 * GPL-Statements eingefuegt
 *
 * Revision 1.3  2002/11/18 15:23:18  joerg
 * Komplett umgebaut
 *
 * Revision 1.2  2002/11/06 11:14:27  joerg
 * getAttributeValues behandelt NamingException selbst.
 *
 * Revision 1.1  2002/11/06 08:20:51  joerg
 * Neue Klassen fuer LDAP-Zugriffe
 *
 */
