/*******************************************************************************
 * Copyright (C) 2002, 2003
 * ingenieurbuero fuer innovative informationstechnik (iiit)
 * Dipl.-Ing. Joerg Beckmann, Dortmund, Germany
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * or look at http://www.gnu.org/copleft/lesser.html.
 *
 * version $Id: LdapConnectionManager.java,v 1.12 2003/01/01 21:03:49 joerg Exp $
 ******************************************************************************/

package de.iiit.ldap;

import de.iiit.xmlconfig.*;

import java.util.*;
import javax.naming.*;
import javax.naming.ldap.*;

import org.apache.log4j.Logger;

/** Manages connection to one or many LDAP servers.
 * @version $Revision: 1.12 $ $Date: 2003/01/01 21:03:49 $
 */
public class LdapConnectionManager
{
    /** CVS Version Tag */
    private static final String vcid = "$Id: LdapConnectionManager.java,v 1.12 2003/01/01 21:03:49 joerg Exp $";
    
    private Logger logger = Logger.getLogger(this.getClass());
    
    private Vector initialContexts = new Vector();
    
    /** Counter for Round-Robin */
    private int nextContext = 0;
    
    private Vector ldapConfigs = new Vector();
    private Vector lostConfigs = new Vector();
    
    private Hashtable masterConfig = null;    
    private InitialLdapContext masterContext = null;
    
    private ContextRefresher refresher = null;

    /** Creates a new instance of LdapConnectionManager */
    private LdapConnectionManager()
    {
    }

    /** Creates a new instance of LdapConnectionManager
     * @param serverConfig a Vector of Configurations
     * @param masterServerNumber the index of the LDAP server with write access in the ServerConfig Vector
     * @param refreshIntervall If a connection to a server crashes a reconnect will be tried after this period
     * of time
     */    
    public LdapConnectionManager(Vector serverConfig, int masterServerNumber, int refreshIntervall)
    {
        int iMax = serverConfig.size();
        
        for (int i = 0; i < iMax; i++)
        {                
            Configuration envCfg = (Configuration) serverConfig.get(i);
            Hashtable env = new Hashtable();
                       
            String url = envCfg.getAttribute(Context.PROVIDER_URL);

            Enumeration keys = envCfg.getAttributeKeys();
            
            while (keys.hasMoreElements())
            {
                String key = (String) keys.nextElement();
                
                if (key.indexOf(".") != 0)
                {
                    env.put(key.toLowerCase(), envCfg.getAttribute(key));
                    logger.info("LDAP config: <" + key.toLowerCase() + "> - <" + envCfg.getAttribute(key) + ">");
                }
            }

            try 
            {
                InitialLdapContext context = new InitialLdapContext(env, null); 
                
                if (i == masterServerNumber)
                {
                    masterConfig = env;
                    masterContext = context;
                    logger.info("LDAP server " + url + " configured as master server");
                }
                else
                {
                    initialContexts.add(context);
                    ldapConfigs.add(env);
                    logger.info("LDAP server " + url + " configured");
                }
            }
            catch (NamingException ne)
            {
                String msg = "Failed to initialize ldap context no. " + i + ": " + ne.getExplanation();

                Throwable t = ne.getRootCause();
		if (t != null)
                {
                    msg += " - " + t.getMessage();
                }
                
                logger.error(msg);

                if (i == masterServerNumber)
                {
                    masterConfig = env;
                }
                else
                {
                    lostConfigs.add(env);
                }
            }
        }
            
        if (masterContext == null)
            logger.warn("No master context configured, switching to read-only mode");

        refresher = new ContextRefresher(refreshIntervall); 
        refresher.start();
    }

    /** Stop the refresher thread 
     */
    public void shutdown()
    {
        refresher.shutdown();
    }
    
    /** Get a javax.naming.ldap.LdapContext
     * @return the requested context or null if none is available
     */    
    public LdapContext getContext() 
    {
        LdapContext result = null;
        
        while (result == null && initialContexts.size() > 0) 
        {
            try
            {
                if (nextContext >= initialContexts.size())
                    nextContext = 0;
            
                InitialLdapContext iContext = (InitialLdapContext) initialContexts.get(nextContext);
                LdapContext ctx = iContext.newInstance(null);
                ctx.reconnect(null);

                result = ctx;
                
                nextContext++;
            }
            catch(NamingException ne)
            {
                logger.error(ne.getMessage(), ne);
                
                Hashtable env = (Hashtable) ldapConfigs.get(nextContext);
                String url = (String) env.get(Context.PROVIDER_URL);
                logger.warn("LDAP server " + url + " lost.");

                synchronized(lostConfigs)
                {
                    ldapConfigs.remove(nextContext);
                    initialContexts.remove(nextContext);
                    
                    lostConfigs.add(env);
                }
            }
        }
            
        if (result == null)
            logger.error("No LDAP server available");
        
        return result;
    }
   
    private class ContextRefresher extends Thread
    {
        private long sleepTime;
        private boolean running = true;
        
        ContextRefresher(long sleepTime)
        {
            this.sleepTime = sleepTime;                        
            setPriority(MIN_PRIORITY);
        }
        
        protected void shutdown()
        {
            running = false;
            interrupt();
        }
        
        public void run()
        {
            while(running)
            {
                int i = 0;
                
                while (i < lostConfigs.size())
                {
                    try
                    {
                        Hashtable env = (Hashtable) lostConfigs.get(i);
                        InitialLdapContext context = new InitialLdapContext(env, null); 
                        context.reconnect(null);
                            
                        synchronized(lostConfigs)
                        {
                            initialContexts.add(context);
                            ldapConfigs.add(env);
                            lostConfigs.remove(i);
                        }

                        String url = (String) env.get(Context.PROVIDER_URL);
                        logger.info("LDAP server " + url + " up again");                        
                    }
                    catch (NamingException ne)
                    {
                        i++;
                    }
                }
                                
                if (masterContext == null && masterConfig != null)
                {
                    try
                    {
                        InitialLdapContext context = new InitialLdapContext(masterConfig, null); 
                            
                        masterContext = context;
                        String url = (String) masterConfig.get(Context.PROVIDER_URL);
                        logger.info("LDAP master server " + url + " up again");                        
                    }
                    catch (NamingException ne)
                    {
                        // Do nothing
                    }
                }
                    
                if (running)
                {
                    try
                    {
                        sleep(sleepTime);
                    }
                    catch(Exception e)
                    {
                        // Do nothing
                    }
                }
            }
        }
    }    
}



/**
 * $Log: LdapConnectionManager.java,v $
 * Revision 1.12  2003/01/01 21:03:49  joerg
 * Copyright-Statement aktualisiert
 *
 * Revision 1.11  2002/12/27 20:17:24  joerg
 * Reconnect verlorener Server korrigiert.
 * Fehlermeldungen verbessert.
 *
 * Revision 1.10  2002/12/27 16:29:56  joerg
 * Alle JNDI-Property-Attribute werden automatisch in das
 * JNDI-Environment uebernommen.
 *
 * Revision 1.9  2002/12/24 21:02:27  joerg
 * Diverse Fehler korrigiert
 *
 * Revision 1.8  2002/12/23 11:26:15  joerg
 * shutdown()-Methode hinzugefuegt.
 *
 * Revision 1.7  2002/12/10 21:53:39  joerg
 * Copyright-Statement korrigiert.
 *
 * Revision 1.6  2002/12/10 11:01:53  joerg
 * JavaDoc ergaenzt
 *
 * Revision 1.5  2002/12/07 20:31:07  joerg
 * Javadoc weiter vervollstaendigt.
 *
 * Revision 1.4  2002/12/07 20:00:00  joerg
 * Namen der LDAP-Configurationsparameter werden jetzt aus
 * dem JNDI-Context uebernommen
 *
 * Revision 1.3  2002/11/30 14:53:43  joerg
 * no message
 *
 * Revision 1.2  2002/11/23 14:21:52  joerg
 * GPL-Statements eingefuegt
 *
 * Revision 1.1  2002/11/06 08:20:51  joerg
 * Neue Klassen fuer LDAP-Zugriffe
 *
 */
