/*******************************************************************************
 * Copyright (C) 2002, 2003
 * ingenieurbuero fuer innovative informationstechnik (iiit)
 * Dipl.-Ing. Joerg Beckmann, Dortmund, Germany
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * or look at http://www.gnu.org/copleft/lesser.html.
 *
 * version $Id: JdbcStatement.java,v 1.6 2003/01/01 21:03:49 joerg Exp $
 ******************************************************************************/


package de.iiit.jdbc;

import java.sql.*;

/** Encapsulates a java.sql.Statement 
 * @version $Revision: 1.6 $ $Date: 2003/01/01 21:03:49 $
 */
public class JdbcStatement
{
    /** CVS Version Tag */
    private static final String vcid = "$Id: JdbcStatement.java,v 1.6 2003/01/01 21:03:49 joerg Exp $";
    
    /** Creates a new instance of JdbcStatement */
    private JdbcConnection connection = null;
    private Statement statement = null;
    
    /** Creates a new instance of JdbcPreparedStatement */
    private JdbcStatement()
    {
    }
    
    /** Creates a new instance of JdbcStatement
     * @param connection the connection that was used to create the meta data object.
     * @param statement the encapsulated java.sql.Statement
     */    
    protected JdbcStatement(JdbcConnection connection, Statement statement)
    {
        this.connection = connection.lock();
        this.statement = statement;
    }
    
    /** Executes the given SQL statement, which may return multiple results.
     * @param sql any SQL statement
     * @throws JdbcException if a database access error occurs
     * @return true if the first result is a ResultSet object; false if it is an update count
     * or there are no results
     */    
    public boolean execute(String sql) throws JdbcException
    {
        try
        {
            return statement.execute(sql);
        }
        catch (SQLException e)
        {
            connection.destroyConnection();
            throw new JdbcException(e);
        }
    }
    
    /** Releases this Statement object's database and JDBC resources immediately instead
     * of waiting for this to happen when it is automatically closed. It is generally
     * good practice to release resources as soon as you are finished with them to avoid
     * tying up database resources.
     * @throws JdbcException if a database access error occurs
     */    
    public void close() throws JdbcException
    {
        try
        {
            statement.close();
            connection.release();
            connection = null;
        }
        catch (SQLException e)
        {
            connection.destroyConnection();
            throw new JdbcException(e);
        }
    }

    /** Release the JdbcConnection hold by this object explicitly.
     * @throws Throwable Any throwable thrown during the excution.
     */    
    protected void finalize() throws Throwable
    {
        if (connection != null)
            connection.release();
    }

}

/**
 * $Log: JdbcStatement.java,v $
 * Revision 1.6  2003/01/01 21:03:49  joerg
 * Copyright-Statement aktualisiert
 *
 * Revision 1.5  2002/12/10 21:53:38  joerg
 * Copyright-Statement korrigiert.
 *
 * Revision 1.4  2002/11/30 14:54:11  joerg
 * Javadoc ergaenzt
 *
 * Revision 1.3  2002/11/23 14:21:52  joerg
 * GPL-Statements eingefuegt
 *
 * Revision 1.2  2002/11/23 14:16:17  joerg
 * JavaDoc-Kommentare eingefuegt
 *
 * Revision 1.1  2002/11/21 21:48:05  joerg
 * Schale um JDBC mit Connection-Pooling.
 * Nur die benoetigten Klassen und methoden sind realisiert.
 *
 */
