/*******************************************************************************
 * Copyright (C) 2002, 2003
 * ingenieurbuero fuer innovative informationstechnik (iiit)
 * Dipl.-Ing. Joerg Beckmann, Dortmund, Germany
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * or look at http://www.gnu.org/copleft/lesser.html.
 *
 * version $Id: JdbcResultSet.java,v 1.6 2003/01/01 21:03:49 joerg Exp $
 ******************************************************************************/

package de.iiit.jdbc;

import java.sql.*;

/** Encapsulates a java.sql.ResultSet 
 * @version $Revision: 1.6 $ $Date: 2003/01/01 21:03:49 $
 */
public class JdbcResultSet
{
    /** CVS Version Tag */
    private static final String vcid = "$Id: JdbcResultSet.java,v 1.6 2003/01/01 21:03:49 joerg Exp $";
    
    private JdbcConnection connection = null;
    private ResultSet resultSet = null;
    
    /** Creates a new instance of JdbcResultSet */
    private JdbcResultSet()
    {
    }
    
    /** Creates a new instance of JdbcResultSet
     * @param connection the connection that was used to create the meta data object.
     * @param resultSet The encapsulated java.sql.ResultSet
     */    
    protected JdbcResultSet(JdbcConnection connection, ResultSet resultSet)
    {
        this.connection = connection.lock();
        this.resultSet = resultSet;
    }
    
    /** Releases this JdbcResultSet object's database and JDBC resources immediately instead
     * of waiting for this to happen when it is automatically closed.
     * @throws JdbcException if a database access error occurs
     *
     */    
    public void close() throws JdbcException
    {
        try
        {
            resultSet.close();
            connection.release();
            connection = null;
        }
        catch (SQLException e)
        {
            connection.destroyConnection();
            throw new JdbcException(e);
        }
    }

    /** Moves the cursor down one row from its current position.
     * @throws JdbcException if a database access error occurs
     *
     * @return true if the new current row is valid; false if there are no more rows
     */    
    public boolean next() throws JdbcException
    {
        try
        {
            return resultSet.next();
        }
        catch (SQLException e)
        {
            connection.destroyConnection();
            throw new JdbcException(e);
        }
    }

    /** Moves the cursor to the first row in this ResultSet object.
     * @throws JdbcException if a database access error occurs or the result set type is TYPE_FORWARD_ONLY
     * @return true if the cursor is on a valid row; false if there are no rows in the result set
     */    
    public boolean first() throws JdbcException
    {
        try
        {
            return resultSet.first();
        }
        catch (SQLException e)
        {
            connection.destroyConnection();
            throw new JdbcException(e);
        }
    }
    
    /** Retrieves the value of the designated column in the current row of this
     * ResultSet object as an int in the Java programming language.
     * @param columnIndex the first column is 1, the second is 2, ...
     * @throws JdbcException if a database access error occurs
     * @return the column value; if the value is SQL NULL, the value returned is 0
     */    
    public int getInt(int columnIndex) throws JdbcException
    {
        try
        {
            return resultSet.getInt(columnIndex);
        }
        catch (SQLException e)
        {
            connection.destroyConnection();
            throw new JdbcException(e);
        }
    }
    
    /** Retrieves the value of the designated column in the current row of this ResultSet
     * object as a long in the Java programming language.
     * @param columnIndex the first column is 1, the second is 2, ...
     * @throws JdbcException if a database access error occurs
     * @return the column value; if the value is SQL NULL, the value returned is 0
     */    
    public long getLong(int columnIndex) throws JdbcException
    {
        try
        {
            return resultSet.getLong(columnIndex);
        }
        catch (SQLException e)
        {
            connection.destroyConnection();
            throw new JdbcException(e);
        }
    }
    
    /** Retrieves the value of the designated column in the current row of this ResultSet
     * object as a long in the Java programming language.
     * @param columnIndex the first column is 1, the second is 2, ...
     * @throws JdbcException if a database access error occurs
     * @return the column value; if the value is SQL NULL, the value returned is null
     */    
    public String getString(int columnIndex) throws JdbcException
    {
        try
        {
            return resultSet.getString(columnIndex);
        }
        catch (SQLException e)
        {
            connection.destroyConnection();
            throw new JdbcException(e);
        }
    }
    
    /** Release the JdbcConnection hold by this object explicitly.
     * @throws Throwable Any throwable thrown during the excution.
     */    
    protected void finalize() throws Throwable
    {
        connection.release();
    }
    
}

/**
 * $Log: JdbcResultSet.java,v $
 * Revision 1.6  2003/01/01 21:03:49  joerg
 * Copyright-Statement aktualisiert
 *
 * Revision 1.5  2002/12/10 21:53:38  joerg
 * Copyright-Statement korrigiert.
 *
 * Revision 1.4  2002/11/30 14:54:11  joerg
 * Javadoc ergaenzt
 *
 * Revision 1.3  2002/11/23 14:21:52  joerg
 * GPL-Statements eingefuegt
 *
 * Revision 1.2  2002/11/23 14:16:17  joerg
 * JavaDoc-Kommentare eingefuegt
 *
 * Revision 1.1  2002/11/21 21:48:05  joerg
 * Schale um JDBC mit Connection-Pooling.
 * Nur die benoetigten Klassen und methoden sind realisiert.
 *
 */
