/*******************************************************************************
 * Copyright (C) 2002, 2003
 * ingenieurbuero fuer innovative informationstechnik (iiit)
 * Dipl.-Ing. Joerg Beckmann, Dortmund, Germany
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * or look at http://www.gnu.org/copleft/lesser.html.
 *
 * version $Id: JdbcPreparedStatement.java,v 1.6 2003/01/01 21:03:49 joerg Exp $
 ******************************************************************************/

package de.iiit.jdbc;

import java.sql.*;

/** Encapsulates a java.sql.PreparedStatement 
 * @version $Revision: 1.6 $ $Date: 2003/01/01 21:03:49 $
 */
public class JdbcPreparedStatement
{
    /** CVS Version Tag */
    private static final String vcid = "$Id: JdbcPreparedStatement.java,v 1.6 2003/01/01 21:03:49 joerg Exp $";
    
    private JdbcConnection connection = null;
    private PreparedStatement statement = null;
    
    /** Creates a new instance of JdbcPreparedStatement */
    private JdbcPreparedStatement()
    {
    }
    
    /** Creates a new instance of JdbcPreparedStatement
     * @param connection the connection that was used to create the meta data object.
     * @param statement the encapsulated java.sql.PreparedStatement
     */    
    protected JdbcPreparedStatement(JdbcConnection connection, PreparedStatement statement)
    {
        this.connection = connection.lock();
        this.statement = statement;
    }
    
    /** Executes the SQL query in this PreparedStatement object and returns the
     * JdbcResultSet object generated by the query.
     * @throws JdbcException if a database access error occurs or the SQL statement does not return a JdbcResultSet object
     * @return a <CODE>JdbcResultSet</CODE> object that contains the data produced by the query; never null
     */    
    public JdbcResultSet executeQuery() throws JdbcException
    {
        try
        {
            return new JdbcResultSet(connection, statement.executeQuery());
        }
        catch (SQLException e)
        {
            connection.destroyConnection();
            throw new JdbcException(e);
        }
    }

    /** Executes the SQL statement in this PreparedStatement object, which must be an
     * SQL INSERT, UPDATE or DELETE statement; or an SQL statement that returns nothing,
     * such as a DDL statement.
     * @throws JdbcException if a database access error occurs or the SQL statement returns a JdbcResultSet object
     *
     * @return either (1) the row count for INSERT, UPDATE, or DELETE statements or (2) 0 for
     * SQL statements that return nothing
     */    
    public int executeUpdate() throws JdbcException
    {
        try
        {
            return statement.executeUpdate();
        }
        catch (SQLException e)
        {
            connection.destroyConnection();
            throw new JdbcException(e);
        }
    }
    
     /** Empties this JdbcPreparedStatement object's current list of SQL commands.
      * @throws JdbcException if a database access error occurs or the driver does not support batch updates
      */    
    public void clearBatch() throws JdbcException
    {
        try
        {
            statement.clearBatch();
        }
        catch (SQLException e)
        {
            connection.destroyConnection();
            throw new JdbcException(e);
        }
    }
    
    /** Clears all the warnings reported on this Statement object. After a call to this
     * method, the method getWarnings will return null until a new warning is reported
     * for this Statement object.
     * @throws JdbcException if a database access error occurs
     */    
    public void clearWarnings() throws JdbcException    
    {
        try
        {
            statement.clearWarnings();
        }
        catch (SQLException e)
        {
            connection.destroyConnection();
            throw new JdbcException(e);
        }
    }
    
    /** Releases this Statement object's database and JDBC resources immediately instead
     * of waiting for this to happen when it is automatically closed. It is generally
     * good practice to release resources as soon as you are finished with them to
     * avoid tying up database resources.
     * @throws JdbcException if a database access error occurs
     */    
    public void close() throws JdbcException
    {
        try
        {
            statement.close();
            connection.release();
            connection = null;
        }
        catch (SQLException e)
        {
            connection.destroyConnection();
            throw new JdbcException(e);
        }
    }

    /** Sets the designated parameter to the given Java String value. The driver converts
     * this to an SQL VARCHAR or LONGVARCHAR value (depending on the argument's size
     * relative to the driver's limits on VARCHAR values) when it sends it to the database.
     * @param parameterIndex the first parameter is 1, the second is 2, ...
     * @param x the parameter value
     * @throws JdbcException if a database access error occurs
     */    
    public void setString(int parameterIndex, String x) throws JdbcException     
    {
        try
        {
            statement.setString(parameterIndex, x);
        }
        catch (SQLException e)
        {
            connection.destroyConnection();
            throw new JdbcException(e);
        }
    }
    
    /** Sets the designated parameter to the given Java long value. The driver converts
     * this to an SQL BIGINT value when it sends it to the database
     * @param parameterIndex the first parameter is 1, the second is 2, ...
     * @param x the parameter value
     * @throws JdbcException if a database access error occurs
     */    
    public void setLong(int parameterIndex, long x) throws JdbcException
    {
        try
        {
            statement.setLong(parameterIndex, x);
        }
        catch (SQLException e)
        {
            connection.destroyConnection();
            throw new JdbcException(e);
        }
    }

    /** Release the JdbcConnection hold by this object explicitly.
     * @throws Throwable Any throwable thrown during the excution.
     */    
    protected void finalize() throws Throwable
    {
        connection.release();
    }

}

/**
 * $Log: JdbcPreparedStatement.java,v $
 * Revision 1.6  2003/01/01 21:03:49  joerg
 * Copyright-Statement aktualisiert
 *
 * Revision 1.5  2002/12/10 21:53:38  joerg
 * Copyright-Statement korrigiert.
 *
 * Revision 1.4  2002/11/30 14:54:11  joerg
 * Javadoc ergaenzt
 *
 * Revision 1.3  2002/11/23 14:21:52  joerg
 * GPL-Statements eingefuegt
 *
 * Revision 1.2  2002/11/23 14:16:17  joerg
 * JavaDoc-Kommentare eingefuegt
 *
 * Revision 1.1  2002/11/21 21:48:05  joerg
 * Schale um JDBC mit Connection-Pooling.
 * Nur die benoetigten Klassen und methoden sind realisiert.
 *
 */
