/*******************************************************************************
 * Copyright (C) 2002, 2003
 * ingenieurbuero fuer innovative informationstechnik (iiit)
 * Dipl.-Ing. Joerg Beckmann, Dortmund, Germany
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * or look at http://www.gnu.org/copleft/lesser.html.
 *
 * version $Id: JdbcDatabaseMetaData.java,v 1.6 2003/01/01 21:03:49 joerg Exp $
 ******************************************************************************/

package de.iiit.jdbc;

import java.sql.*;

/** Encapsulates an java.sql.DatabaseMetaData 
 * @version $Revision: 1.6 $ $Date: 2003/01/01 21:03:49 $
 */
public class JdbcDatabaseMetaData
{
    /** CVS Version Tag */
    private static final String vcid = "$Id: JdbcDatabaseMetaData.java,v 1.6 2003/01/01 21:03:49 joerg Exp $";
    
    private JdbcConnection connection = null;
    private DatabaseMetaData metaData = null;

    /** Creates a new instance of JdbcDatabaseMetaData */
    private JdbcDatabaseMetaData()
    {
    }
    
    /** Creates a new instance of JdbcDatabaseMetaData
     * @param connection the connection that was used to create the meta data object.
     * @param metaData the encapsulated DataBaseMetaData
     */    
    protected JdbcDatabaseMetaData(JdbcConnection connection, DatabaseMetaData metaData)
    {
        this.connection = connection.lock();
        this.metaData = metaData;
    }
    
    /** Retrieves a description of the tables available in the given catalog. Only table
     * descriptions matching the catalog, schema, table name and type criteria are returned.
     * They are ordered by TABLE_TYPE, TABLE_SCHEM and TABLE_NAME.
     * @param catalog a catalog name; must match the catalog name as it is stored in the database; ""
     * retrieves those without a catalog; null means that the catalog name should not
     * be used to narrow the search
     * @param schemaPattern a schema name pattern; must match the schema name as it is stored in the
     * database; "" retrieves those without a schema; null means that the schema name
     * should not be used to narrow the search
     * @param tableNamePattern a table name pattern; must match the table name as it is stored in the database
     * @param types a list of table types to include; null returns all types
     * @throws JdbcException if a database access error occurs
     * @return each row is a table description
     */    
    public JdbcResultSet getTables(String catalog, String schemaPattern, String tableNamePattern, String[] types) throws JdbcException
    {
        try
        {
            return new JdbcResultSet(connection, metaData.getTables(catalog, schemaPattern, tableNamePattern, types));
        }
        catch (SQLException e)
        {
            connection.destroyConnection();
            throw new JdbcException(e);
        }
    }
    
    /** Releases this Statement object's database and JDBC resources immediately instead
     * of waiting for this to happen when it is automatically closed. It is generally
     * good practice to release resources as soon as you are finished with them to
     * avoid tying up database resources.
     */    
    public void close()
    {
        connection.release();
        connection = null;
    }

    /** Release the JdbcConnection hold by this object explicitly.
     * @throws Throwable Any throwable thrown during the excution.
     */    
    protected void finalize() throws Throwable
    {
        connection.release();
    }

}

/**
 * $Log: JdbcDatabaseMetaData.java,v $
 * Revision 1.6  2003/01/01 21:03:49  joerg
 * Copyright-Statement aktualisiert
 *
 * Revision 1.5  2002/12/10 21:53:38  joerg
 * Copyright-Statement korrigiert.
 *
 * Revision 1.4  2002/11/30 14:54:11  joerg
 * Javadoc ergaenzt
 *
 * Revision 1.3  2002/11/23 14:21:52  joerg
 * GPL-Statements eingefuegt
 *
 * Revision 1.2  2002/11/23 14:16:17  joerg
 * JavaDoc-Kommentare eingefuegt
 *
 * Revision 1.1  2002/11/21 21:48:05  joerg
 * Schale um JDBC mit Connection-Pooling.
 * Nur die benoetigten Klassen und methoden sind realisiert.
 *
 */
