/*******************************************************************************
 * Copyright (C) 2002, 2003
 * ingenieurbuero fuer innovative informationstechnik (iiit)
 * Dipl.-Ing. Joerg Beckmann, Dortmund, Germany
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * or look at http://www.gnu.org/copleft/lesser.html.
 *
 * version $Id: JdbcConnectionPool.java,v 1.9 2003/04/02 09:25:08 joerg Exp $
 ******************************************************************************/

package de.iiit.jdbc;

import org.apache.log4j.Logger;

import java.util.*;
import java.sql.*;

/** To be used as an pooled replacement of java.sql.connection. 
 * @version $Revision: 1.9 $ $Date: 2003/04/02 09:25:08 $
 */
public class JdbcConnectionPool
{
    /** CVS Version Tag */
    private static final String vcid = "$Id: JdbcConnectionPool.java,v 1.9 2003/04/02 09:25:08 joerg Exp $";
    
    private String url      = null;
    private String user     = null;
    private String password = null;
    
    private Vector freeConnections = new Vector();
    private Vector busyConnections = new Vector();

    private Logger logger = Logger.getLogger(this.getClass());
    
    /** Creates a new instance of JdbcConnectionPool */
    private JdbcConnectionPool()
    {
    }

    /** Creates a pool prepared to connect to the given database URL.
     * @param url a database url of the form jdbc:<I>subprotocol</I>:<I>subname</I>
     * @param user the database user on whose behalf the connection is being made
     * @param password the user's password
     */    
    public JdbcConnectionPool(String url, String user, String password)
    {
        this.url      = url;
        this.user     = user;
        this.password = password;
    }
    
    /** Creates a pool prepared to connect to the given database URL and opens 
     * pre-defined connectios for later use.
     * @param url a database url of the form jdbc:<I>subprotocol</I>:<I>subname</I>
     * @param user the database user on whose behalf the connection is being made
     * @param password the user's password
     * @param connections then number of connections to open
     */    
    public JdbcConnectionPool(String url, String user, String password, int connections)
    {
        this(url, user, password);
        
        logger.info("Creating " + connections + " new connections to DB <" + url + ">.");

        if (connections > 0)
        {
            try
            {
                for (int i = 0; i < connections; i++)
                {
                    freeConnections.add(new JdbcConnection(this, DriverManager.getConnection(url, user, password)));
                }
            }
            catch(SQLException e)
            {
                logger.error("Caught SQLException while creating connections", e);
            }

            logger.info(freeConnections.size() + " new connection to DB <" + url + "> created.");
        }
    }
    
    /** Creates a Statement object for sending SQL statements to the database.
     * @throws JdbcException if a database access error occurs
     * @return a new default JdbcStatement object
     */    
    public JdbcStatement createStatement() throws JdbcException
    {
        return getLockedConnection().createStatement();
    }
    
    /** Creates a PreparedStatement object for sending parameterized SQL statements to the database.
     * @param sql an SQL statement that may contain one or more '?' IN parameter placeholders
     * @throws JdbcException if a database access error occurs
     * @return a new default JdbcPreparedStatement object containing the pre-compiled SQL statement
     */    
    public JdbcPreparedStatement prepareStatement(String sql) throws JdbcException
    {
        return getLockedConnection().prepareStatement(sql);
    }
    
    /** Retrieves a JdbcDatabaseMetaData object that contains metadata about the database
     * to which this JdbcConnectionPool object represents a connection.
     * @throws JdbcException if a database access error occurs
     * @return a DatabaseMetaData object for this Connection object
     */    
    public JdbcDatabaseMetaData getMetaData() throws JdbcException
    {
        return getLockedConnection().getMetaData();
    }
    
    private JdbcConnection getLockedConnection() throws JdbcException
    {
        JdbcConnection result = null;
        
        synchronized(busyConnections)
        {
            if (freeConnections.isEmpty())
            {
                try
                {
                    result = new JdbcConnection(this, DriverManager.getConnection(url, user, password));
                    logger.info("New connection to DB <" + url + "> created, having " + (busyConnections.size() + 1));
                }
                catch(SQLException e)
                {
                    throw new JdbcException(e);
                }
            }
            else
            {
                result = (JdbcConnection) freeConnections.firstElement();
                freeConnections.remove(result);
            }

            busyConnections.add(result);
        }
        
        return result;
    }
    
    /** marks the connection as <I>free</I> and moves it into the pool of free
     * connections.
     * @param connection The JdbcConnection to release
     */    
    protected void releaseConnection(JdbcConnection connection)
    {
        synchronized(busyConnections)
        {
            if (busyConnections.contains(connection))
            {
                busyConnections.remove(connection);
                freeConnections.add(connection);        
            }
        }
    }
    
    /** Remove the connection from the pool of busy ones but do not move it to free
     * ones. This method is called by JdbcConnection.destroyConnection() in case on an
     * error.
     * @param connection the connection to remove.
     */    
    protected void destroyConnection(JdbcConnection connection)
    {
        synchronized(busyConnections)
        {
            if (busyConnections.contains(connection))
                busyConnections.remove(connection);
        }
    }
}

/**
 * $Log: JdbcConnectionPool.java,v $
 * Revision 1.9  2003/04/02 09:25:08  joerg
 * New constructor.
 *
 * Revision 1.8  2003/03/12 13:23:42  joerg
 * Log messages added to show the creation of new connections.
 *
 * Revision 1.7  2003/01/01 21:03:49  joerg
 * Copyright-Statement aktualisiert
 *
 * Revision 1.6  2002/12/10 21:53:38  joerg
 * Copyright-Statement korrigiert.
 *
 * Revision 1.5  2002/11/30 14:54:11  joerg
 * Javadoc ergaenzt
 *
 * Revision 1.4  2002/11/26 14:44:06  joerg
 * Fehler bei synchronize behoben
 *
 * Revision 1.3  2002/11/23 14:21:52  joerg
 * GPL-Statements eingefuegt
 *
 * Revision 1.2  2002/11/23 14:16:17  joerg
 * JavaDoc-Kommentare eingefuegt
 *
 * Revision 1.1  2002/11/21 21:48:06  joerg
 * Schale um JDBC mit Connection-Pooling.
 * Nur die benoetigten Klassen und methoden sind realisiert.
 *
 */
