/*******************************************************************************
 * Copyright (C) 2002, 2003
 * ingenieurbuero fuer innovative informationstechnik (iiit)
 * Dipl.-Ing. Joerg Beckmann, Dortmund, Germany
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * or look at http://www.gnu.org/copleft/lesser.html.
 *
 * version $Id: JdbcConnection.java,v 1.6 2003/01/01 21:03:49 joerg Exp $
 ******************************************************************************/

package de.iiit.jdbc;

import java.sql.*;

/** This class holds a <CODE>java.sql.connection</CODE>. It should be used only inside
 * this package. Therefore all methods are protected.
 * @version $Revision: 1.6 $ $Date: 2003/01/01 21:03:49 $
 */
public class JdbcConnection
{
    /** CVS Version Tag */
    private static final String vcid = "$Id: JdbcConnection.java,v 1.6 2003/01/01 21:03:49 joerg Exp $";
    
    private JdbcConnectionPool owner;
    
    private Connection connection;
    
    private int refCounter = 0;
    
    /** Creates a new instance of JdbcConnection */
    private JdbcConnection()
    {
    }

    /** Creates a new instance of JdbcConnection
     * @param owner The owning JdbcConnectionPool
     * @param connection The connection to the underlying database
     */    
    protected JdbcConnection(JdbcConnectionPool owner, Connection connection)
    {
        this.owner      = owner;
        this.connection = connection;
    }
    
    /** Lock a collection. Every object except JdbcConnectionPools must lock a
     * JdbcConnection a long as it uses it. As long as there are any locks on the
     * Connection it is marked 'busy'.
     * @return This.
     */    
    protected JdbcConnection lock()
    {
        refCounter++;
        
        return this;
    }
    
    /** Release a previous lock. When the the last lock on the connection is released
     * the connection will be marked 'free' and is returned to the pool of free
     * connections to be reused.
     */    
    protected void release()
    {
        if (--refCounter == 0)
            owner.releaseConnection(this);            
    }
    
    /** Creates prepared statement
     * @param sql an SQL statement that may contain one or more '?' IN parameter placeholders
     * @throws JdbcException if a database access error occurs
     * @return a new default JdbcPreparedStatement object containing the pre-compiled SQL statement
     */    
    protected JdbcPreparedStatement prepareStatement(String sql) throws JdbcException
    {
        try
        {
            return new JdbcPreparedStatement(this, connection.prepareStatement(sql));
        }
        catch (SQLException e)
        {
            owner.destroyConnection(this);
            
            throw new JdbcException(e);
        }
    }
    
    /** Creates a Statement object for sending SQL statements to the database.
     * @throws JdbcException if a database access error occurs
     * @return a new default Statement object
     */    
    protected JdbcStatement createStatement() throws JdbcException
    {
        try
        {
            return new JdbcStatement(this, connection.createStatement());
        }
        catch (SQLException e)
        {
            owner.destroyConnection(this);
            
            throw new JdbcException(e);
        }
    }

    /** Retrieves a <CODE>JdbcDatabaseMetaData</CODE> object that contains metadata about
     * the database to which this Connection object represents a connection.
     * @throws JdbcException if a database access error occurs
     * @return a JdbcDatabaseMetaData object for this Connection object
     */    
    protected JdbcDatabaseMetaData getMetaData() throws JdbcException
    {
        try
        {
            return new JdbcDatabaseMetaData(this, connection.getMetaData());
        }
        catch (SQLException e)
        {
            owner.destroyConnection(this);
            
            throw new JdbcException(e);
        }
    }
    
    /** Do not move the connection to pool of free ones. This method must be called in
     * case of an catched SQLException.
     */    
    protected void destroyConnection()
    {
        owner.destroyConnection(this);
    }
    
    /** closes the encapsulated java.sql.Connection explicitly.
     * @throws Throwable Any throwable thrown during the excution.
     */    
    protected void finalize() throws Throwable
    {
        try
        {
            connection.close();
        }
        catch (SQLException e)
        {
            // do nothing;
        }
    }
}

/**
 * $Log: JdbcConnection.java,v $
 * Revision 1.6  2003/01/01 21:03:49  joerg
 * Copyright-Statement aktualisiert
 *
 * Revision 1.5  2002/12/10 21:53:38  joerg
 * Copyright-Statement korrigiert.
 *
 * Revision 1.4  2002/11/30 14:54:11  joerg
 * Javadoc ergaenzt
 *
 * Revision 1.3  2002/11/23 14:21:52  joerg
 * GPL-Statements eingefuegt
 *
 * Revision 1.2  2002/11/23 14:16:17  joerg
 * JavaDoc-Kommentare eingefuegt
 *
 * Revision 1.1  2002/11/21 21:48:05  joerg
 * Schale um JDBC mit Connection-Pooling.
 * Nur die benoetigten Klassen und methoden sind realisiert.
 *
 */
