/*******************************************************************************
 * Copyright (C) 2002, 2003
 * ingenieurbuero fuer innovative informationstechnik (iiit)
 * Dipl.-Ing. Joerg Beckmann, Dortmund, Germany
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * or look at http://www.gnu.org/copleft/lesser.html.
 *
 * version $Id: GenericTimeoutCache.java,v 1.12 2003/04/14 20:38:02 joerg Exp $
 ******************************************************************************/

package de.iiit.cache;

import java.util.*;

import org.apache.log4j.Logger;

/** A cache with background clean-up thread which removes all entries not used for a
 * given period of time.
 * @version $Revision: 1.12 $ $Date: 2003/04/14 20:38:02 $
 */
public class GenericTimeoutCache extends GenericCache
{
    /** CVS Version Tag */
    private static final String vcid = "$Id: GenericTimeoutCache.java,v 1.12 2003/04/14 20:38:02 joerg Exp $";
    
    private long timeout = -1L;
    
    private long sleeptime = 1000L;
    
    private CacheCleaner cleaner;

    private Logger logger = Logger.getLogger(this.getClass());
    
    /** Creates a new instance of GenericTimeoutCache */
    public GenericTimeoutCache()
    {
        this(-1L, 1000L);
    }

    /** Creates a new instance of GenericTimeoutCache
     * @param timeout The period of time after which unused entries shall be removed automatically
     */    
    public GenericTimeoutCache(long timeout)
    {
        this(timeout, 1000L);
    }
    
    /** Creates a new instance of GenericTimeoutCache
     * @param timeout The period of time after which unused entries shall be removed automatically
     * @param sleeptime The amount of time the clean-up thread shall sleep between to runs.
     */    
    public GenericTimeoutCache(long timeout, long sleeptime)
    {
        this.timeout = timeout;

        if (timeout >= 0L)
        {
            cleaner = new CacheCleaner(sleeptime);
            cleaner.start();
        }
    }
        
    /** Stop the cleaner thread 
     */
    public void shutdown()
    {
        cleaner.shutdown();
    }
    
    /** Add a new element to the cache. If an element with the same key already exists
     * it will be replaced.
     * @param key The unique key of the cache entry
     * @param value The value of the cache entry
     */    
    public void addElement(Object key, Object value)
    {
        GenericTimeoutCacheElement ce = new GenericTimeoutCacheElement(timeout, value);
        super.addElement(key, ce);
    }
    
    /** Retrieves an cache-entry. The timeout of the cache element is reset by this
     * method.
     * @param key The unique key of the cache entry to retrieve
     * @throws CacheFaultException if there is no entry with the given key
     * @return The cache entry with the given key
     */    
    public Object getElement(Object key) throws CacheFaultException
    {
        GenericTimeoutCacheElement ce = (GenericTimeoutCacheElement) super.getElement(key);
        ce.setTimeout(timeout);
        
        return ce.getValue();
    }
    
    /** Retrieves the timeout of the cache element. The timeout of the cache element
     * will not be changed by this method.
     * @param key The unique key of the cache entry to retrieve
     * @return The timeout of the cache element or -1L if there is no element with the given key.
     */    
    protected long getElementTimeout(Object key)
    {
        try
        {
            GenericTimeoutCacheElement ce = (GenericTimeoutCacheElement) super.getElement(key);
            return ce.getTimeout();
        }
        catch(CacheFaultException e)
        {
            return -1L;
        }
    }
    
    private class CacheCleaner extends Thread
    {
        private long sleepTime;
        private boolean running = true;
    
        CacheCleaner(long sleepTime)
        {
            setName("CacheCleaner");

            this.sleepTime = sleepTime;
            
            setPriority(MIN_PRIORITY);
        }

        protected void shutdown()
        {
            running = false;
            interrupt();
        }
        
        public void run()
        {
            while(running)
            {
                try 
                {
                    for (Enumeration keys = getKeys(); keys.hasMoreElements(); ) 
                    {
                        Object k = keys.nextElement();
                        long t = getElementTimeout(k);
                        if (t > 0 && t < System.currentTimeMillis())
                        {
                            logger.info("Timeout reached for key <" + k.toString() + ">");
                            removeElement(k);
                        }
                        
                        if (running)
                            sleep(sleepTime);
                    }
                    if (running)
                        sleep(sleepTime);
                }
                catch(Exception e)
                {
                }
            }
            
            running = false;
        }
    }    

    private class GenericTimeoutCacheElement
    {    
        private long timeout = -1L;
    
        private Object value;
    
        private GenericTimeoutCacheElement()
        {
        }
    
        protected GenericTimeoutCacheElement(long timeout)
        {
            if (timeout >= 0L)
                this.timeout = System.currentTimeMillis() + timeout;
        }
    
        protected GenericTimeoutCacheElement(long timeout, Object value)
        {
            if (timeout >= 0L)
                this.timeout = System.currentTimeMillis() + timeout;
        
            this.value = value;
        }
    
        protected void setTimeout(long timeout)
        {
            if (timeout >= 0L)
                this.timeout = System.currentTimeMillis() + timeout;
            else
                this.timeout = -1L;
        }
    
        protected long getTimeout()
        {
            return timeout;
        }
    
        protected Object getValue()
        {
            return value;
        }
    }

}

/**
 * $Log: GenericTimeoutCache.java,v $
 * Revision 1.12  2003/04/14 20:38:02  joerg
 * Setting name for thread.
 *
 * Revision 1.11  2003/01/01 21:03:49  joerg
 * Copyright-Statement aktualisiert
 *
 * Revision 1.10  2002/12/22 20:40:46  joerg
 * shutdown()-Methode eingebaut
 *
 * Revision 1.9  2002/12/10 21:53:38  joerg
 * Copyright-Statement korrigiert.
 *
 * Revision 1.8  2002/12/10 11:01:53  joerg
 * JavaDoc ergaenzt
 *
 * Revision 1.7  2002/12/10 10:05:23  joerg
 * CacheElement-Klassen als private Klassen in die entsprechenden
 * Cache-Klassen aufgenommen
 *
 * Revision 1.6  2002/11/30 14:54:11  joerg
 * Javadoc ergaenzt
 *
 * Revision 1.5  2002/11/23 14:21:52  joerg
 * GPL-Statements eingefuegt
 *
 * Revision 1.4  2002/11/23 14:16:28  joerg
 * JavaDoc-Kommentare eingefuegt
 *
 * Revision 1.3  2002/11/21 21:49:00  joerg
 * Ausgaben auf logger umgestellt.
 *
 * Revision 1.2  2002/11/07 21:05:09  joerg
 * Anbindung des CacheCleaners geaendert
 *
 * Revision 1.1  2002/11/01 21:36:47  joerg
 * Version mit funktionierenden 1st-level Cache
 *
 */
