/*******************************************************************************
 * Copyright (C) 2002, 2003
 * ingenieurbuero fuer innovative informationstechnik (iiit)
 * Dipl.-Ing. Joerg Beckmann, Dortmund, Germany
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * or look at http://www.gnu.org/copleft/lesser.html.
 *
 * version $Id: GenericInvalidationCache.java,v 1.7 2003/01/01 21:03:49 joerg Exp $
 ******************************************************************************/

package de.iiit.cache;

import java.util.*;

/** A cache with a guaranteed maximum lifetime of each entry 
 * @version $Revision: 1.7 $ $Date: 2003/01/01 21:03:49 $
 */
public class GenericInvalidationCache extends GenericCache
{
    /** CVS Version Tag */
    private static final String vcid = "$Id: GenericInvalidationCache.java,v 1.7 2003/01/01 21:03:49 joerg Exp $";
    
    private long invalidationTimeout = -1L;

    /** Creates a new instance of GenericInvalidationCache */
    public GenericInvalidationCache()
    {
    }

    /** Creates a new instance of GenericInvalidationCache
     * @param invalidationTimeout The maximum lifetime of the entries
     */    
    public GenericInvalidationCache(long invalidationTimeout)
    {
        this.invalidationTimeout = invalidationTimeout;
    }
    
    /** Add a new element to the cache. If an element with the same key already exists
     * it will be replaced.
     * @param key The unique key of the cache entry
     * @param value The unique key of the cache entry
     */    
    public void addElement(Object key, Object value)
    {
        CacheElement ce = new CacheElement(invalidationTimeout, value);
        super.addElement(key, ce);
    }
    
    /** Retrieves an cache-entry. If the cache entry is older than the maximum lifetime
     * it will be removed instead.
     * @param key The unique key of the cache entry to retrieve
     * @throws CacheFaultException if there is no entry with the given key or if the entry is to old.
     * @return The cache entry with the given key
     */    
    public Object getElement(Object key) throws CacheFaultException
    {
        CacheElement ce = (CacheElement) super.getElement(key);
        long t = ce.getInvalidationTime();
        
        if (t > 0L && t < System.currentTimeMillis())
        {
            removeElement(key);
            throw new CacheFaultException();
        }
        else
            return ce.getValue();
    }

    private class CacheElement
    {
        private long invalidationTime = -1L;
    
        private Object value;
    
        private CacheElement()
        {
        }
    
        protected CacheElement(long invalidationTimeout)
        {
            if (invalidationTimeout >= 0L)
                invalidationTime = System.currentTimeMillis() + invalidationTimeout;
        }
    
        protected CacheElement(long invalidationTimeout, Object value)
        {
            if (invalidationTimeout >= 0L)
                invalidationTime = System.currentTimeMillis() + invalidationTimeout;
        
            this.value = value;
        }
    
        protected long getInvalidationTime()
        {
            return invalidationTime;
        }
    
        protected Object getValue()
        {
            return value;
        }
    }

}

/**
 * $Log: GenericInvalidationCache.java,v $
 * Revision 1.7  2003/01/01 21:03:49  joerg
 * Copyright-Statement aktualisiert
 *
 * Revision 1.6  2002/12/10 21:53:38  joerg
 * Copyright-Statement korrigiert.
 *
 * Revision 1.5  2002/12/10 10:05:23  joerg
 * CacheElement-Klassen als private Klassen in die entsprechenden
 * Cache-Klassen aufgenommen
 *
 * Revision 1.4  2002/11/30 14:54:11  joerg
 * Javadoc ergaenzt
 *
 * Revision 1.3  2002/11/23 14:21:52  joerg
 * GPL-Statements eingefuegt
 *
 * Revision 1.2  2002/11/23 14:16:28  joerg
 * JavaDoc-Kommentare eingefuegt
 *
 * Revision 1.1  2002/11/01 21:36:47  joerg
 * Version mit funktionierenden 1st-level Cache
 *
 */
